/* Measure Octave accuracy (on E1433) */

#include <stdlib.h>             /* For exit */
#include <stdio.h>              /* For printf */
#include <string.h>             /* For strcmp, strcat */
#include <math.h>               /* For sqrt */

/* #define VXIPNP */ /* uncomment or -D it for PnP build */

#ifdef  VXIPNP
#include "hpe1432.h"
#else   /* VXIPNP */
#include "e1432.h"
#endif  /* VXIPNP */

#ifndef	M_PI	/* not everyone seems to have this defined */
#define M_PI	3.14159265358979323846	/* out of math.h */
#endif

/* constants for filter specs */
#define OCT_FILT_ORD_THIRD	3
#define OCT_FILT_ORD_FULL	7
#define QD_MIN			.944
#define QD_MAX			1.059

#ifdef  VXIPNP
#define LIB_STATUS      ViStatus
#define LIB_HW          ViSession
#define LIB_GROUP       ViInt32
#define LIB_ID          ViInt32
#define LIB_ENUM        ViInt32
#define LIB_INT         ViInt32
#define LIB_LONG        ViInt32
#define LIB_FLOAT       ViReal64
#else   /* VXIPNP */
#define LIB_STATUS      SHORTSIZ16
#define LIB_HW          E1432ID
#define LIB_GROUP       SHORTSIZ16
#define LIB_ID          SHORTSIZ16
#define LIB_ENUM        SHORTSIZ16
#define LIB_INT         SHORTSIZ16
#define LIB_LONG        LONGSIZ32
#define LIB_FLOAT       FLOATSIZ32
#endif  /* VXIPNP */

#ifdef  VXIPNP
#define HWCONF_SPACE                    27
#define HWCONF_ID                       0
#define HWCONF_ID_OCT                   0x1000
#define HWCONF_IN_CHANS                 21
#define HWCONF_SRC_CHANS                22
#define BLOCKSIZE_DEF                   1024
#define INPUT_CHANS                     16
#define SRC_CHANS                       5
#define AVG_RMS                         HPE1432_AVG_RMS
#define BLOCK_MODE                      HPE1432_BLOCK_MODE
#define CHANNEL_OFF                     HPE1432_CHANNEL_OFF
#define CHANNEL_ON                      HPE1432_CHANNEL_ON
#define COUPLING_AC                     HPE1432_COUPLING_AC
#define COUPLING_DC                     HPE1432_COUPLING_DC
#define CONTINUOUS_MODE                 HPE1432_CONTINUOUS_MODE
#define DATA_MODE_OVERLAP_FREERUN       HPE1432_DATA_MODE_OVERLAP_FREERUN
#define DATA_MODE_OVERLAP_BLOCK         HPE1432_DATA_MODE_OVERLAP_BLOCK
#define DATA_SIZE_16                    HPE1432_DATA_SIZE_16
#define DATA_SIZE_32                    HPE1432_DATA_SIZE_32
#define DATA_SIZE_32_SERV               HPE1432_DATA_SIZE_32_SERV
#define DATA_SIZE_FLOAT32               HPE1432_DATA_SIZE_FLOAT32
#define DATA_SIZE_DEF                   HPE1432_DATA_SIZE_16
#define DATA_FREQ                       HPE1432_DATA_FREQ
#define DATA_TIME                       HPE1432_DATA_TIME
#define FREQ_DATA                       HPE1432_FREQ_DATA
#define OCTAVE_DATA                     HPE1432_OCTAVE_DATA
#define ENABLE_OFF                      HPE1432_ENABLE_OFF
#define ENABLE_ON                       HPE1432_ENABLE_ON
#define ENABLE_TYPE_TIME                HPE1432_ENABLE_TYPE_TIME
#define ENABLE_TYPE_FREQ                HPE1432_ENABLE_TYPE_FREQ
#define ENABLE_TYPE_OCTAVE              HPE1432_ENABLE_TYPE_OCTAVE
#define INPUT_MODE_VOLT                 HPE1432_INPUT_MODE_VOLT
#define INPUT_HIGH_NORMAL               HPE1432_INPUT_HIGH_NORMAL
#define ANTI_ALIAS_ANALOG_ON            HPE1432_ANTI_ALIAS_ANALOG_ON
#define OCTAVE_MEAS_OFF                 HPE1432_OCTAVE_MEAS_OFF
#define OCTAVE_MEAS_ON                  HPE1432_OCTAVE_MEAS_ON
#define OCTAVE_AVG_MODE_LIN             HPE1432_OCTAVE_AVG_MODE_LIN
#define OCTAVE_AVG_MODE_EXP             HPE1432_OCTAVE_AVG_MODE_EXP
#define OCTAVE_AVG_MODE_DEF             HPE1432_OCTAVE_AVG_MODE_DEF
#define OCTAVE_HOLD_MODE_OFF            HPE1432_OCTAVE_HOLD_MODE_OFF
#define OCTAVE_HOLD_MODE_MIN            HPE1432_OCTAVE_HOLD_MODE_MIN
#define OCTAVE_HOLD_MODE_MAX            HPE1432_OCTAVE_HOLD_MODE_MAX
#define OCTAVE_HOLD_MODE_DEF            HPE1432_OCTAVE_HOLD_MODE_DEF
#define OCTAVE_MODE_FULL                HPE1432_OCTAVE_MODE_FULL
#define OCTAVE_MODE_THIRD               HPE1432_OCTAVE_MODE_THIRD
#define OCTAVE_MODE_DEF                 HPE1432_OCTAVE_MODE_DEF
#define OCTAVE_TIME_CONST_DEF           HPE1432_OCTAVE_TIME_CONST_DEF
#define OCTAVE_INT_TIME_DEF             HPE1432_OCTAVE_INT_TIME_DEF
#define OCTAVE_TIME_STEP_DEF            HPE1432_OCTAVE_TIME_STEP_DEF
#define WINDOW_FLATTOP                  HPE1432_WINDOW_FLATTOP
#define WEIGHTING_OFF                   HPE1432_WEIGHTING_OFF

#else   /* VXIPNP */

#define BLOCKSIZE_DEF                   E1432_BLOCKSIZE_DEF
#define INPUT_CHANS                     E1432_INPUT_CHANS
#define SRC_CHANS                       E1432_SRC_CHANS
#define AVG_RMS                         E1432_AVG_RMS
#define BLOCK_MODE                      E1432_BLOCK_MODE
#define CHANNEL_OFF                     E1432_CHANNEL_OFF
#define CHANNEL_ON                      E1432_CHANNEL_ON
#define COUPLING_AC                     E1432_COUPLING_AC
#define COUPLING_DC                     E1432_COUPLING_DC
#define CONTINUOUS_MODE                 E1432_CONTINUOUS_MODE
#define DATA_MODE_OVERLAP_FREERUN       E1432_DATA_MODE_OVERLAP_FREERUN
#define DATA_MODE_OVERLAP_BLOCK         E1432_DATA_MODE_OVERLAP_BLOCK
#define DATA_SIZE_16                    E1432_DATA_SIZE_16
#define DATA_SIZE_32                    E1432_DATA_SIZE_32
#define DATA_SIZE_32_SERV               E1432_DATA_SIZE_32_SERV
#define DATA_SIZE_FLOAT32               E1432_DATA_SIZE_FLOAT32
#define DATA_SIZE_DEF                   E1432_DATA_SIZE_DEF
#define DATA_FREQ                       E1432_DATA_FREQ
#define DATA_TIME                       E1432_DATA_TIME
#define FREQ_DATA                       E1432_FREQ_DATA
#define OCTAVE_DATA                     E1432_OCTAVE_DATA
#define ENABLE_OFF                      E1432_ENABLE_OFF
#define ENABLE_ON                       E1432_ENABLE_ON
#define ENABLE_TYPE_TIME                E1432_ENABLE_TYPE_TIME
#define ENABLE_TYPE_FREQ                E1432_ENABLE_TYPE_FREQ
#define ENABLE_TYPE_OCTAVE              E1432_ENABLE_TYPE_OCTAVE
#define INPUT_MODE_VOLT                 E1432_INPUT_MODE_VOLT
#define INPUT_HIGH_NORMAL               E1432_INPUT_HIGH_NORMAL
#define ANTI_ALIAS_ANALOG_ON            E1432_ANTI_ALIAS_ANALOG_ON
#define OCTAVE_MEAS_OFF                 E1432_OCTAVE_MEAS_OFF
#define OCTAVE_MEAS_ON                  E1432_OCTAVE_MEAS_ON
#define OCTAVE_AVG_MODE_LIN             E1432_OCTAVE_AVG_MODE_LIN
#define OCTAVE_AVG_MODE_EXP             E1432_OCTAVE_AVG_MODE_EXP
#define OCTAVE_AVG_MODE_DEF             E1432_OCTAVE_AVG_MODE_DEF
#define OCTAVE_HOLD_MODE_OFF            E1432_OCTAVE_HOLD_MODE_OFF
#define OCTAVE_HOLD_MODE_MIN            E1432_OCTAVE_HOLD_MODE_MIN
#define OCTAVE_HOLD_MODE_MAX            E1432_OCTAVE_HOLD_MODE_MAX
#define OCTAVE_HOLD_MODE_DEF            E1432_OCTAVE_HOLD_MODE_DEF
#define OCTAVE_MODE_FULL                E1432_OCTAVE_MODE_FULL
#define OCTAVE_MODE_THIRD               E1432_OCTAVE_MODE_THIRD
#define OCTAVE_MODE_DEF                 E1432_OCTAVE_MODE_DEF
#define OCTAVE_TIME_CONST_DEF           E1432_OCTAVE_TIME_CONST_DEF
#define OCTAVE_INT_TIME_DEF             E1432_OCTAVE_INT_TIME_DEF
#define OCTAVE_TIME_STEP_DEF            E1432_OCTAVE_TIME_STEP_DEF
#define WEIGHTING_OFF                   E1432_WEIGHTING_OFF
#define WINDOW_FLATTOP                  E1432_WINDOW_FLATTOP
#endif  /* VXIPNP */

#ifdef  VXIPNP
#define E1432_INPUT_CHAN(s)     s
#define E1432_SOURCE_CHAN(s)    (4096 + s)
#define e1432_init_measure              hpe1432_initMeasure
#define e1432_reset_measure             hpe1432_resetMeasure
#define e1432_set_active                hpe1432_setActive
#define e1432_set_amp_scale             hpe1432_setAmpScale
#define e1432_set_analog_input          hpe1432_setAnalogInput
#define e1432_set_avg_mode              hpe1432_setAvgMode
#define e1432_set_avg_number            hpe1432_setAvgNumber
#define e1432_get_blocksize             hpe1432_getBlocksize
#define e1432_set_blocksize             hpe1432_setBlocksize
#define e1432_set_calc_data             hpe1432_setCalcData
#define e1432_set_clock_freq            hpe1432_setClockFreq
#define e1432_set_coupling              hpe1432_setCoupling
#define e1432_set_coupling_freq         hpe1432_setCouplingFreq
#define e1432_get_data_mode             hpe1432_getDataMode
#define e1432_set_data_mode             hpe1432_setDataMode
#define e1432_set_data_size             hpe1432_setDataSize
#define e1432_set_enable                hpe1432_setEnable
#define e1432_set_filter_freq           hpe1432_setFilterFreq
#define e1432_get_filter_settling_time  hpe1432_getFilterSettlingTime
#define e1432_set_filter_settling_time  hpe1432_setFilterSettlingTime
#define e1432_get_octave_avg_mode       hpe1432_getOctaveAvgMode
#define e1432_set_octave_avg_mode       hpe1432_setOctaveAvgMode
#define e1432_get_octave_hold_mode      hpe1432_getOctaveHoldMode
#define e1432_set_octave_hold_mode      hpe1432_setOctaveHoldMode
#define e1432_get_octave_int_time       hpe1432_getOctaveIntTime
#define e1432_set_octave_int_time       hpe1432_setOctaveIntTime
#define e1432_set_octave_meas           hpe1432_setOctaveMeas
#define e1432_get_octave_mode           hpe1432_getOctaveMode
#define e1432_set_octave_mode           hpe1432_setOctaveMode
#define e1432_get_octave_start_freq     hpe1432_getOctaveStartFreq
#define e1432_set_octave_start_freq     hpe1432_setOctaveStartFreq
#define e1432_set_octave_stop_freq      hpe1432_setOctaveStopFreq
#define e1432_get_octave_stop_freq      hpe1432_getOctaveStopFreq
#define e1432_get_octave_time_const     hpe1432_getOctaveTimeConst
#define e1432_set_octave_time_const     hpe1432_setOctaveTimeConst
#define e1432_set_octave_time_step      hpe1432_setOctaveTimeStep
#define e1432_get_octave_time_step      hpe1432_getOctaveTimeStep
#define e1432_set_ramp_rate             hpe1432_setRampRate
#define e1432_get_range                 hpe1432_getRange
#define e1432_set_range                 hpe1432_setRange
#define e1432_set_sine_freq             hpe1432_setSineFreq
#define e1432_set_span                  hpe1432_setSpan
#define e1432_get_span                  hpe1432_getSpan
#define e1432_set_weighting             hpe1432_setWeighting
#define e1432_set_window                hpe1432_setWindow
#else   /* VXIPNP */
#endif  /* VXIPNP */

#define BA_TIMEOUT 20000000      /* about 10 minute, S700 hpux, mxi */

#define MAX_LAS         2       /* max number of modules */
#define MAX_SRCS        (MAX_LAS * SRC_CHANS)
#define MAX_CHANS       (MAX_LAS * INPUT_CHANS)

#define MAX_BLOCKSIZE   16384
#define SAMPLE_FREQ     65536.0
#define MAX_BANDS       39      /* max number of octave bands */

#define MIN_FFT_AVGS    1
/*#define MIN_OCT_AVGS    1*/
#define MIN_OCT_AVGS    1

#define MAX_FREQ_PTS    1000
#define MAX_AMP_LIN_PTS 1000
#define MAX_DECAY_PTS   1000

#define SRC_FILTER_FREQ_MAX     25600
#define SRC_FILTER_FREQ_MIN     6400

/* octave band settling times are these numbers / band */
#define FULL_OCT_SETTLING       7.1
#define THIRD_OCT_SETTLING      21.6

#define EXP_SETTLING            10.0

#define DEBUG(s)        s

/* Wrap this around all the many function calls which might fail */
#ifdef  VXIPNP
char vistr[100], details[100];
#define PRINT_ERR(_ver) \
{\
    hpe1432_error_message(hw,_ver,vistr);\
    hpe1432_errorDetails(hw,details,100);\
    printf("error %d = %s, line %d\nDetails: %s\n",_ver,vistr,__LINE__,details);\
}
#define CHECK(func)     \
do {\
    ViStatus _ver;\
    _ver = (func);\
    if (_ver)\
    {\
        printf("%s;\n", #func);\
        DEBUG(PRINT_ERR(_ver))\
        return _ver;\
    }\
} while (0)

#else   /* VXIPNP */

#define DEBUG(s)        s
#ifdef  __lint
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (func)
#else
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (0)
#endif
#endif  /* VXIPNP */


#define FILTER_FREQ(f) \
    ((f < .8 * SRC_FILTER_FREQ_MIN) ? SRC_FILTER_FREQ_MIN : SRC_FILTER_FREQ_MAX)

LIB_FLOAT third_octaves[] =
{
     3.15,   4.0,    5.0,     6.3,     8.0,    10.0,    12.5,    16.0,
    20.0,   25.0,   31.5,    40.0,    50.0,    63.0,    80.0,   100.0,
   125.0,  160.0,  200.0,   250.0,   315.0,   400.0,   500.0,   630.0,
   800.0, 1000.0, 1250.0,  1600.0,  2000.0,  2500.0,  3150.0,  4000.0,
  5000.0, 6300.0, 8000.0, 10000.0, 12500.0, 16000.0, 20000.0
};
int third_bands = sizeof(third_octaves) / sizeof (LIB_FLOAT);


/* enums for command line parm entry */

LIB_ENUM octave_mode_enums[] =
{
    OCTAVE_MODE_FULL, OCTAVE_MODE_FULL,
    OCTAVE_MODE_THIRD, OCTAVE_MODE_THIRD
};
char *octave_mode_strs[] = {
    "F", "f",
    "T", "t",
    NULL
};

LIB_ENUM octave_avg_mode_enums[] =
{
    OCTAVE_AVG_MODE_LIN, OCTAVE_AVG_MODE_LIN,
    OCTAVE_AVG_MODE_EXP, OCTAVE_AVG_MODE_EXP
};
char *octave_avg_mode_strs[] =
{
    "L", "l",
    "E", "e",
    NULL
};

LIB_ENUM octave_hold_mode_enums[] =
{
    OCTAVE_HOLD_MODE_OFF, OCTAVE_HOLD_MODE_OFF,
    OCTAVE_HOLD_MODE_MIN, OCTAVE_HOLD_MODE_MIN,
    OCTAVE_HOLD_MODE_MIN, OCTAVE_HOLD_MODE_MIN,
    OCTAVE_HOLD_MODE_MAX, OCTAVE_HOLD_MODE_MAX
};
char *octave_hold_mode_strs[] =
{
    "O", "o",
    "M", "m",
    "N", "n",
    "X", "x",
    NULL
};

LIB_ENUM data_mode_enums[] =
{
    BLOCK_MODE,                BLOCK_MODE,
    CONTINUOUS_MODE,           CONTINUOUS_MODE,
    DATA_MODE_OVERLAP_FREERUN, DATA_MODE_OVERLAP_FREERUN,
    DATA_MODE_OVERLAP_BLOCK,   DATA_MODE_OVERLAP_BLOCK };
char *data_mode_strs[] =
{
    "b", "B",
    "c", "C",
    "f", "F",
    "o", "O",
    NULL
};

LIB_ENUM data_size_enums[] =
{
    DATA_SIZE_16,
    DATA_SIZE_32_SERV, DATA_SIZE_32_SERV,
    DATA_SIZE_FLOAT32, DATA_SIZE_FLOAT32,
    DATA_SIZE_32  /* last so will not also match 32s, etc */
};
char *data_size_strs[] =
{
    "16",
    "32s", "32S",
    "f32", "F32",
    "32",
    NULL
};


const char reg_meas_err[] =
  "\nOctave regression test error on chan %d%s, %.3g Hz band%s\n  read %lg dB";

/* global state variables */

LIB_FLOAT test_freq = 1000.0;

LIB_FLOAT amp_step = 1.0;
LIB_FLOAT amp_start = 0.0;

/* default cal, meas times, bin size */
LIB_FLOAT fft_meas_time = 2.0;
LIB_FLOAT oct_meas_time = 10.0;
LIB_FLOAT max_cal_meas_time = 0.0; /*sec*/
LIB_FLOAT min_input_cycles = 0.0;

int verbose = 0;
int print_raw = 0;

#ifdef  VXIPNP
#if defined(MSDOS) || defined(WIN32)
char * optarg = NULL;
long optind = 0;
long opterr = 0;

int getopt( int  argc, char **argv, char *optstring )
{
    int opt;

    if (argv != NULL)
    {
        optarg = NULL;
        if (++optind == (long)argc)
        {
            return( EOF );
        }

        if ((argv[optind][0] != '-') && (argv[optind][0] != '/'))
        {
            optarg = argv[optind];
            return( EOF );
        }

        opt = argv[optind][1];

        while (optstring[0] && (optstring[0] != (char) opt)) ++optstring;
        if (!optstring[0]) return( opt );

        if (optstring[1] == ':' || optstring[1] == ';')
        {
            if (argv[optind][2])
            {
                optarg = &argv[optind][2];
            }
            else if (optstring[1] == ';')  /* optional argument */
                return opt;
            else
            {
                if (++optind == (long)argc)
                {
                    return( EOF );
                }

                optarg = argv[optind];
            }
            if (*optarg == ':')
            {
                optarg++;
            }
            else if (*optarg == '=')
            {
                optarg++;
            }
        }
    }
    return( opt );
}
#else   /* MSDOS || WIN32 */
extern char *optarg;
#endif  /* MSDOS || WIN32 */
#endif  /* VXIPNP */


#define VALID_OPTS      \
  "a:b:c:d:e:f:h:i:l:m:n:o:p:r:s:t:uvzA:B:C:D:F:G:H:L:M:RS:T:U"


void
print_usage(char *name)
{
#   if OCTAVE_MODE_DEF == OCTAVE_MODE_FULL
        char md = 'F';
#   else
        char md = 'T';
#   endif

#   if OCTAVE_AVG_MODE_DEF == OCTAVE_AVG_MODE_LIN
        char ad = 'L';
#   else
        char ad = 'E';
#   endif

#   if OCTAVE_HOLD_MODE_DEF == OCTAVE_HOLD_MODE_MAX
        char hd = 'X';
#   endif
#   if OCTAVE_HOLD_MODE_DEF == OCTAVE_HOLD_MODE_MIN
        char hd = 'N';
#   endif
#   if OCTAVE_HOLD_MODE_DEF == OCTAVE_HOLD_MODE_OFF
        char hd = 'O';
#   endif

    (void) fprintf(stdout,"usage: %s [options] file\n",name);
    (void) fprintf(stdout,"options:\n");
    (void) fprintf(stdout,"    -a octave_avg_mode [Lin, Exp]"
      " (default %c)\n", ad);
    (void) fprintf(stdout,"    -b time blocksize"
      " (default %g)\n", BLOCKSIZE_DEF);
    (void) fprintf(stdout,"    -c octave_time_const"
      " (default %g)\n", OCTAVE_TIME_CONST_DEF);
    (void) fprintf(stdout,"    -d max_cal_meas_time"
      " (default %g=disabled)\n", max_cal_meas_time);
    (void) fprintf(stdout,"    -e min_input_cycles"
      " (default %g=disabled)\n", min_input_cycles);
    (void) fprintf(stdout,"    -f test frequency"
      " (default %g)\n", test_freq);
    (void) fprintf(stdout,"    -h octave_hold_mode [maX, miN, Off]"
      " (default %c)\n", hd);
    (void) fprintf(stdout,"    -i octave_int_time"
      " (default %g)\n", OCTAVE_INT_TIME_DEF);
    (void) fprintf(stdout,"    -l amp_lin_err_list"
      " (to test amplitude linearity)\n");
    (void) fprintf(stdout,"    -m data_mode (b, o, f, or c)\n");
    (void) fprintf(stdout,"    -n times (run test multiple times)\n");
    (void) fprintf(stdout,"    -o octave_mode [Full, Third]"
      " (default %c)\n", md);
    (void) fprintf(stdout,"    -p like -l but IEC1260 format\n");
    (void) fprintf(stdout,"       \"lin_error, lin_range, max_error, max_range\"\n");
    (void) fprintf(stdout,"    -r range(volts peak)\n");
    (void) fprintf(stdout,"    -s data_size [16, 32, 32S, F32]\n");
    (void) fprintf(stdout,"    -t octave_time_step"
      " (default %g)\n", OCTAVE_TIME_STEP_DEF);
    (void) fprintf(stdout,"    -v (verbose)\n");
    (void) fprintf(stdout,"    -z (compare to ideal filter)\n");
    (void) fprintf(stdout,"    -A coupling_freq (default none)\n");
    (void) fprintf(stdout,"    -B start/stop bands (to test band response)\n");
    (void) fprintf(stdout,"       or single, override band for -F option\n");
    (void) fprintf(stdout,"    -C cal meas time"
      " (default %.1f sec)\n", fft_meas_time);
    (void) fprintf(stdout,"    -D percent decay errs (to test exp decay)\n");
    (void) fprintf(stdout,"    -F frequencies (to test flatness)\n");
    (void) fprintf(stdout,"    -G as in -F but \"start stop delta\"\n");
    (void) fprintf(stdout,"    -H as in -G but IEC 1260 logartimic frequencies"
      " \"fm, S, N\"\n");
    (void) fprintf(stdout,"    -L logical_address (default 8)\n");
    (void) fprintf(stdout,"    -M Octave meas time"
      " (default %.1f sec)\n", oct_meas_time);
    (void) fprintf(stdout,"    -R print raw data\n");
    (void) fprintf(stdout,"    -S amplitude_step"
      " (default %.2f dB)\n", amp_step);
    (void) fprintf(stdout,"    -T amplitude_start"
      " (default %.2f dBFS)\n", amp_start);
    (void) fprintf(stdout,"    -U (to use FS octave meas for amp lin cal)\n");
}


void
parm_err(char *arg, char *parm_name)
{
    (void) fprintf(stderr, "parameter error: %s \"%s\" not converted\n",
      parm_name, arg);
    exit(2);
}


LIB_LONG
get_long(char *arg, char *parm_name)
{
    double ftmp;
    LIB_LONG lval = 0;
    int rtn;

    /* strip leading "0x", if there */
    if ( strncmp(arg, "0x", 2) == 0 )
    {
        arg += 2;
        rtn = sscanf(arg, "%lx", &lval);
    }
    else /* try float => long conversion next */
    {
        rtn = sscanf(arg, "%lf", &ftmp);
        if ( rtn == 1 )
        {
            lval = (LIB_LONG)(ftmp + .5);
        }
        else /* try hex as a last ditch effort */
        {
            rtn = sscanf(arg, "%lx", &lval);
        }
    }
    if ( rtn != 1 ) parm_err(arg, parm_name);

    return lval;
}


LIB_INT
get_int(char *arg, char *parm_name)
{
    LIB_LONG ltmp = get_long(arg, parm_name);
    return (LIB_INT)ltmp;
}


LIB_FLOAT
get_float(char *arg, char *parm_name)
{
    double ftmp;
    int rtn = sscanf(arg, "%lf", &ftmp);

    if ( rtn != 1 ) parm_err(arg, parm_name);
    return (LIB_FLOAT)ftmp;
}


int
get_float_array(char *arg, char *parm_name, LIB_FLOAT *array, int max)
{
    int len = strlen(arg);
    int i = 0;
    int conv;
    int read = 0;
    double tmp;

    while ( i < len )
    {
        while ( i < len && arg[i] == ' ' ) i++; /* skip blanks */
        if ( i < len )
        {
            conv = sscanf(&arg[i], "%lf", &tmp);
            if ( conv > 0 )
            {
                if ( read >= max && i < len )
                {
                    fprintf(stderr, "exceeded maximum number of points"
                      " (%d) for %s\n", max, parm_name);
                    break;
                }
                *array++ = (LIB_FLOAT)tmp;
                read++;
            }
            else
            {
                break;
            }
        }
        while ( i < len && arg[i] != ' ' ) i++; /* skip what just read */
    }

    if ( i != len ) parm_err(arg, parm_name);

    return read;
}


LIB_ENUM
get_enum(char *arg, LIB_ENUM *enums, char **strs, char *parm_name)
{
    while ( *strs != NULL )
    {
        if ( strcmp(arg, *strs++) == 0 ) return *enums;
        enums++;
    }
    parm_err(arg, parm_name);
}


LIB_STATUS print_parms(LIB_HW hw, LIB_GROUP group)
{
    LIB_STATUS err = 0;
    int i;
    char *oct_str, *avg_str, *hold_str, *dm;
    LIB_ENUM data_mode;
    LIB_ENUM octave_mode, octave_avg_mode, octave_hold_mode;
    LIB_FLOAT octave_int_time, octave_time_const, octave_time_step;
    LIB_LONG blocksize;

    CHECK(e1432_get_blocksize(hw, group, &blocksize));
    CHECK(e1432_get_data_mode(hw, group, &data_mode));
    for ( i = 0; data_mode_enums[i] != data_mode ; i++ );
    dm = data_mode_strs[i];
    (void) printf("  blocksize = 0x%x, data_mode = %s, test_freq = %g\n",
      blocksize, dm, test_freq);

    CHECK(e1432_get_octave_mode(hw, group, &octave_mode));
    if ( octave_mode == OCTAVE_MODE_FULL ) oct_str = "full";
    else if ( octave_mode == OCTAVE_MODE_THIRD ) oct_str = "third";
    else oct_str = "undefined";
    CHECK(e1432_get_octave_avg_mode(hw, group, &octave_avg_mode));
    if ( octave_avg_mode == OCTAVE_AVG_MODE_LIN ) avg_str = "lin";
    else if ( octave_avg_mode == OCTAVE_AVG_MODE_EXP ) avg_str ="exp";
    else avg_str = "undefined";
    CHECK(e1432_get_octave_hold_mode(hw, group, &octave_hold_mode));
    if ( octave_hold_mode == OCTAVE_HOLD_MODE_OFF ) hold_str = "off";
    else if ( octave_hold_mode == OCTAVE_HOLD_MODE_MAX ) hold_str = "max";
    else if ( octave_hold_mode == OCTAVE_HOLD_MODE_MIN ) hold_str = "min";
    else hold_str = "undefined";
    (void) printf("  octave_mode = %s, octave_avg_mode = %s,"
      " octave_hold_mode = %s\n", oct_str, avg_str, hold_str);

    CHECK(e1432_get_octave_int_time(hw, group, &octave_int_time));
    CHECK(e1432_get_octave_time_const(hw, group, &octave_time_const));
    CHECK(e1432_get_octave_time_step(hw, group, &octave_time_step));
    if ( octave_avg_mode == OCTAVE_AVG_MODE_EXP )
    {
        (void) printf("  octave_time_const = %g", octave_time_const);
    }
    else
    {
        (void) printf("  octave_int_time = %g", octave_int_time);
    }
    (void) printf(", octave_time_step = %g\n", octave_time_step);
    (void) printf("  max_cal_meas_time = %g", max_cal_meas_time);
    (void) printf(", min_input_cycles = %g\n", min_input_cycles);

    (void) printf("\n");

    (void) fflush(NULL);

    return err;
}


/* return the number of bands beteen start, stop, inclusive */
LIB_LONG
oct_bands(LIB_HW hw, LIB_GROUP in_group, LIB_FLOAT octave_start_freq,
  LIB_FLOAT octave_stop_freq)
{
    double log_ratio;
    double log_band;
    LIB_ENUM octave_mode = OCTAVE_MODE_THIRD;

    (void)e1432_get_octave_mode(hw, in_group, &octave_mode);

    log_ratio = log(octave_stop_freq / octave_start_freq);
    log_band = log(2);
    if ( octave_mode == OCTAVE_MODE_THIRD ) log_band /= 3.0;

    /* round to nearest + 1 (ratio == 0 => 1 band) */
    return (LIB_LONG)(log_ratio / log_band + 1.5);
}

LIB_FLOAT
oct_band_step(LIB_HW hw, LIB_GROUP in_group)
{
    LIB_ENUM octave_mode = OCTAVE_MODE_THIRD;
    LIB_FLOAT band_step = 2.0;

    (void)e1432_get_octave_mode(hw, in_group, &octave_mode);

    if ( octave_mode == OCTAVE_MODE_THIRD ) band_step = pow(band_step, 1/3.0);

    return band_step;
}


LIB_FLOAT
octave_band(LIB_FLOAT freq)
{
    /*like third_oct, but returns nearest full octave band*/
    int i;
    
    freq /= pow(2.0, 1/2.0);  /* bump down by 1/2 octave band for slop */

    for ( i = 1; i < third_bands - 2 && freq > third_octaves[i]; i=i+3 );

    return third_octaves[i];
}

LIB_FLOAT
third_oct_band(LIB_FLOAT freq)
{
    int i = 0;
    
    freq /= pow(2.0, 1/6.0);  /* bump down by 1/2 third octave band for slop */

    for ( i == 0; i < third_bands - 1 && freq > third_octaves[i]; i++ );

    return third_octaves[i];
}

double
exact_fm(LIB_FLOAT freq, LIB_ENUM octave_mode)
{
    /*returns the closest exact mid band frequency to freq(IEC1260 table A.1)*/
    
    int i;
    double G, fm, x, exp;
    
    G = 2.0;
        
    /*find the index of the closest nominal midband in third_octaves[]*/
    if ( octave_mode == OCTAVE_MODE_THIRD ) 
    {
      freq /= pow(2.0, 1/6.0);  /* bump down by 1/2 third octave band for slop */
      for ( i = 0; i < third_bands - 1 && freq > third_octaves[i]; i++ );    
    }
    else if ( octave_mode == OCTAVE_MODE_FULL )
    {
      freq /= pow(2.0, 1/2.0);  /* bump down by 1/2 octave band for slop */
      for ( i = 1; i < third_bands - 2 && freq > third_octaves[i]; i=i+3 );
    }
    else
    {
       (void) fprintf(stderr, "octave mode(%d) not defined.\n", octave_mode);
       exit(2);
    }
    
    /*calculate the index x (table A.1)*/
    x = (double)i - 25.0;
    
    /*calculate the exact midband frequency*/
    exp = x/3.0;
    fm = 1000.0*pow(2.0, exp);
    
    return fm;
}

double
ref_nbw(LIB_FLOAT freq, LIB_ENUM octave_mode)
{
    /*returns the closest reference bandwidth(IEC1260) for a frequency*/
    
    double b, bw, G, exp, fm;
        
    G = 2.0;
    
    /*find the exact mid band frequency*/
    fm = exact_fm(freq, octave_mode);
        
    if ( octave_mode == OCTAVE_MODE_FULL ) 
    {
      b = 1.0;
    }
    else if ( octave_mode == OCTAVE_MODE_THIRD )
    {
      b = 3.0;
    }
    else
    {
       (void) fprintf(stderr, "octave mode(%d) not defined.\n", octave_mode);
       exit(2);
    }

    exp = 1.0/(2.0*b);
    bw = fm*(pow(G,exp) - pow(G,-1*exp));
    
    return bw;
}


LIB_STATUS
wait_block_available(LIB_HW hw, LIB_GROUP group, long loop)
#ifdef  VXIPNP
{ return 0; }   /* no block_available in PnP */
#else   /* VXIPNP */
{
    LIB_STATUS status;
    char *error_string;
    long ba_count = 0;

    do  /* Wait for block available and check for errors and warnings  */
    {
        status = e1432_block_available(hw, group);
        error_string = e1432_get_error_string();
    } while ( status != 1 && ba_count++ < BA_TIMEOUT
      && *error_string == NULL );
    if ( status < 0 || *error_string != NULL )
    {
        (void) fprintf(stderr, "block_available() error \"%s\", loop %d\n",
          error_string, loop);
        if ( status < 0 ) return status;
        return -1;
    }
    if ( ba_count >= BA_TIMEOUT )
    {
        (void) fprintf(stderr, "block_available() timeout, loop %d\n",
          loop);
        return -1;
    }
    return 0;
}
#endif  /* VXIPNP */


void
s_subt(double *from, double subt, double *result, int N)
{
    int i;
    for ( i = 0; i < N; i++ )
    {
        *result++ = *from++ - subt;
    }
}


void
v_subt(double *from, double *subt, double *result, int N)
{
    int i;
    for ( i = 0; i < N; i++ )
    {
        *result++ = *from++ - *subt++;
    }
}


void
s_div(double *from, double div, double *result, int N)
{
    int i;
    for ( i = 0; i < N; i++ )
    {
        *result++ = *from++ / div;
    }
}


void
s_mult(double *from, double mult, double *result, int N)
{
    int i;
    for ( i = 0; i < N; i++ )
    {
        *result++ = *from++ * mult;
    }
}


void
s_min_max(double *min, double *max, double *data, int N)
{
    int i;

    /* initialize with first data point */
    *min = *data;
    *max = *data++;

    /* do the remainder of the data points */
    for ( i = 1; i < N; i++ )
    {
        if ( *data > *max ) *max = *data;
        else if ( *data < *min ) *min = *data;
        data++;
    }
}


void
v_min_max(double *min, double *max, double **data, int M, int N)
{
    int i, j;
    double *dptr, *nptr, *xptr;

    /* initialize max/min with first channel */
    dptr = *data++;
    nptr = min;
    xptr = max;
    for ( j = 0; j < N; j++ )
    {
        *nptr++ = *dptr;
        *xptr++ = *dptr++;
    }

    /* remainder of channels */
    for ( i = 1; i < M; i++ )
    {
        dptr = *data++;
        nptr = min;
        xptr = max;
        for ( j = 0; j < N; j++ )
        {
            if ( *dptr > *xptr ) *xptr = *dptr;
            else if ( *dptr < *nptr ) *nptr = *dptr;
            dptr++;
            nptr++;
            xptr++;
        }
    }
}


void
s_avg(double *avg, double *data, int N)
{
    int i;
    *avg = 0;
    for ( i = 0; i < N; i++ )
    {
        *avg += *data++;
    }
    *avg /= N;
}

void
v_avg(double *avg, double **data, int M, int N)
{
    int i, j;
    double *dptr, *aptr;

    /* clear average */
    aptr = avg;
    for ( j = 0; j < N; j++ )
    {
        *aptr++ = 0;
    }

    /* accumulate */
    for ( i = 0; i < M; i++ )
    {
        dptr = *data++;
        aptr = avg;
        for ( j = 0; j < N; j++ )
        {
            *aptr++ += *dptr++;
        }
    }

    /* normalize */
    aptr = avg;
    for ( j = 0; j < N; j++ )
    {
        *aptr++ /= (double)M;
    }
}


double dB(double pwr)
{
    return 10.0 * log10(pwr);
}

void
print_doubles(double *data, int N, char *fmt)
{
    int i;
    for ( i = 0; i < N; i++ )
    {
        printf(fmt, *data++);
    }
    printf("\n");
}

void
print_bands(char *title, double *data, LIB_LONG bands,
  LIB_FLOAT octave_start_freq, LIB_FLOAT band_step)
{
    int i;
    LIB_FLOAT band = octave_start_freq;

    printf("%s\n", title);

    for ( i = 0; i < bands; i++ )
    {
        if ( i % 6 == 0 )
        {
            if ( i > 0 ) printf("\n");
            printf("  %8.2lf  ", (double)third_oct_band(band));
        }
        printf("%9.3lf", (double)(*data++));
        band *= band_step;
    }
    printf("\n");
}

void
print_chan_nums(LIB_ID *chan_list, LIB_INT chans, char *title, char *fmt)
{
    int i;
    printf(title);
    for ( i = 0; i < chans; i++ )
    {
        printf(fmt, *chan_list++);
    }
    printf("\n");
}


void
gen_ideal(LIB_FLOAT freq, double Q_factor, LIB_FLOAT octave_start_freq, LIB_ENUM octave_mode, double *response, LIB_LONG bands)
{
    int octOrd;
    int band;
    double filtOrd;	/* filter order */
    double octN;	/* band number, relative to 1k band */
    double center;
    double Qr, Qd;
    double tmp;

    if ( octave_mode == OCTAVE_MODE_THIRD )
    {
        octOrd = 3;	/* 1/3 Octave */
        filtOrd = OCT_FILT_ORD_THIRD;
    }
    else
    {
        octOrd = 1;	/* 1/1 Octave */
        filtOrd = OCT_FILT_ORD_FULL;
    }

    /* use the power of 2 system, since that's how the filters are designed */
    /* the 1000 offset is to insure that the rounding is done on a positive
       number, the .5 is for rounding to nearest int */
    octN = (double)((int)(1000.5 + log(octave_start_freq /1000) / log(2)
      * octOrd) - 1000);
    tmp = pow(2.0, .5 / octOrd);
    Qr = 1.0 / (tmp - 1.0/tmp);
    tmp = .5 * M_PI / filtOrd;
    Qd = Qr * tmp / sin(tmp);
    Qd *= Q_factor;
    for ( band = octN; band < octN + bands; band++ )
    {
	center = 1000 * pow(2.0, (double)band / octOrd);
        tmp = center / freq;
	tmp = tmp - 1.0 / tmp;
	if ( tmp < 0 ) tmp = -tmp;
        *response++ = -10.0 * log10(1.0 + pow(Qd * tmp, 2.0 * filtOrd));
    }
}


LIB_STATUS
set_filter_settling_time(LIB_HW hw, LIB_GROUP in_group, LIB_FLOAT freq)
{
    LIB_FLOAT filter_settling_time;
    double band_settle_factor;
    LIB_ENUM octave_mode;
    LIB_ENUM octave_avg_mode;

    /* band filter settling */
    CHECK(e1432_get_octave_mode(hw, in_group, &octave_mode));
    band_settle_factor = (octave_mode == OCTAVE_MODE_THIRD) ?
      THIRD_OCT_SETTLING : FULL_OCT_SETTLING;
    filter_settling_time = band_settle_factor / freq;

    /* exp avg settling */
    CHECK(e1432_get_octave_avg_mode(hw, in_group, &octave_avg_mode));
    if ( octave_avg_mode == OCTAVE_AVG_MODE_EXP )
    {
        LIB_FLOAT octave_time_const;
        CHECK(e1432_get_octave_time_const(hw, in_group, &octave_time_const));
        filter_settling_time += EXP_SETTLING * octave_time_const;
    }

    CHECK(e1432_set_filter_settling_time(hw, in_group, filter_settling_time));
}


LIB_STATUS
read_f64_data(LIB_HW hw, LIB_ID *in_chan_list, LIB_INT in_chans,
  LIB_ENUM data_type, double *data, LIB_LONG blocksize)
{
    int i;
    LIB_LONG count;
#ifdef  VXIPNP
    ViInt32 wait = 1;
#else   /* VXIPNP */
    struct e1432_trailer trailer;
#endif  /* VXIPNP */

    for (i = 0; i < in_chans; i++ )
    {
#ifdef  VXIPNP
        CHECK(hpe1432_readFloat64Data(hw, *in_chan_list++, data_type,
          data, blocksize, &count, wait));
#else   /* VXIPNP */
        CHECK(e1432_read_float64_data(hw, *in_chan_list++, data_type,
          data, blocksize, &trailer, &count));
#endif  /* VXIPNP */
        *data++ = 10 * log10(*data);
    }

    return 0;
}


LIB_STATUS
fft_meas(LIB_HW hw, LIB_GROUP in_group, LIB_INT in_chans,
  LIB_ID *in_chan_list, double *results, LIB_FLOAT freq,
  LIB_FLOAT meas_time)
{
#ifdef  VXIPNP
    ViInt32 wait = 1;
#else   /* VXIPNP */
    struct e1432_trailer trailer;
#endif  /* VXIPNP */
    LIB_LONG blocksize;
    LIB_LONG count;
    double *freq_data;
    LIB_FLOAT span, old_span;
    LIB_FLOAT save_filter_settling_time;
    int fft_bin;
    LIB_INT avg_number;
    int i;

    /* malloc freq data array */
    CHECK(e1432_get_blocksize(hw, in_group, &blocksize));
    freq_data = (double *)malloc(sizeof(double) * blocksize);

    /* save span, filter_settling_time */
    CHECK(e1432_get_span(hw, in_group, &old_span));
    CHECK(e1432_get_filter_settling_time(hw, in_group,
      &save_filter_settling_time));

    /* fft meas setup */
    CHECK(e1432_set_calc_data(hw, in_group, DATA_FREQ));
    CHECK(e1432_set_window(hw, in_group, WINDOW_FLATTOP));
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_TIME,
      ENABLE_OFF));
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_FREQ,
      ENABLE_ON));
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_OCTAVE,
      ENABLE_OFF));

    CHECK(e1432_set_filter_settling_time(hw, in_group, .1));

    /* span, avg, fft_bin setup */
    span = freq;
    if ( blocksize / (2.56 * span) > meas_time )
    {
        if ( max_cal_meas_time <= 0.0 )
        {
           span = blocksize / (2.56 * meas_time);
        }
        else
        {
           if ( blocksize / (2.56 * span) > max_cal_meas_time )
           {
              span = blocksize / (2.56 * max_cal_meas_time);
           }
           else
           {
              /* do nothing, leave span=freq */
           }
        }
    }
    /*printf("span=%g, blocksize=%d, meas_time=%g\n", span, blocksize, meas_time);*/
    CHECK(e1432_set_span(hw, in_group, span));
    CHECK(e1432_get_span(hw, in_group, &span));
    avg_number = (LIB_INT)(meas_time / (blocksize / (2.56f * span)));
    /*printf("avg_number=%d\n", avg_number);*/
    if ( avg_number < MIN_FFT_AVGS ) avg_number = (LIB_INT)MIN_FFT_AVGS;
    fft_bin = (int)(freq * (LIB_FLOAT)blocksize / 2.56 / span + .5);
    CHECK(e1432_set_avg_mode(hw, in_group, AVG_RMS));
    CHECK(e1432_set_avg_number(hw, in_group, avg_number));

    /* Start measurement, wait for data */
    CHECK(e1432_init_measure(hw, in_group));
    CHECK(wait_block_available(hw, in_group, 0));

    for (i = 0; i < in_chans; i++ )
    {
#ifdef  VXIPNP
        CHECK(hpe1432_readFloat64Data(hw, in_chan_list[i], FREQ_DATA,
          freq_data, blocksize/2, &count, wait));
#else   /* VXIPNP */
        CHECK(e1432_read_float64_data(hw, in_chan_list[i], FREQ_DATA,
          freq_data, blocksize/2, &trailer, &count));
#endif  /* VXIPNP */
        *results++ = 10 * log10(.5*freq_data[fft_bin]);
    }

    /* clean up and restore */
    CHECK(e1432_reset_measure(hw, in_group));
    CHECK(e1432_set_calc_data(hw, in_group, DATA_TIME));
    CHECK(e1432_set_span(hw, in_group, old_span));
    CHECK(e1432_set_filter_settling_time(hw, in_group,
      save_filter_settling_time));

    free(freq_data);

    return 0;
}


LIB_STATUS
oct_meas(LIB_HW hw, LIB_GROUP in_group, LIB_INT in_chans, LIB_ID *in_chan_list,
  LIB_LONG bands, double **results, LIB_FLOAT meas_time)
{
#ifdef  VXIPNP
    ViInt32 wait = 1;
#else   /* VXIPNP */
    struct e1432_trailer trailer;
#endif  /* VXIPNP */
    double oct_buf[40];  /* mallocing this causes problems in hpux 10.20!?! */
    LIB_LONG count;
    double *dptr, *doptr;
    double **rptr;
    LIB_FLOAT octave_time_step;
    int i, j;
    int avg, avgs;

    /* compute number of averages */
    CHECK(e1432_get_octave_time_step(hw, in_group, &octave_time_step));
    avgs = (int)(meas_time / octave_time_step);
    if ( avgs < MIN_OCT_AVGS ) avgs = MIN_OCT_AVGS;

    /* enable only Octave data */
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_TIME,
      ENABLE_OFF));
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_FREQ,
      ENABLE_OFF));
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_OCTAVE,
      ENABLE_ON));

    /* clear the average buffer */
    rptr = results;
    for (i = 0; i < in_chans; i++ )
    {
        dptr = *rptr++;
        for ( j = 0; j < bands; j++ )
        {
            *dptr++ = 0;
        }
    }

    /* Start measurement */
    CHECK(e1432_init_measure(hw, in_group));

    for ( avg = 0; avg < avgs; avg++ )
    {
        rptr = results;
        CHECK(wait_block_available(hw, in_group, 0));

        for (i = 0; i < in_chans; i++ )
        {
#ifdef  VXIPNP
            CHECK(hpe1432_readFloat64Data(hw, in_chan_list[i], OCTAVE_DATA,
              oct_buf, bands, &count, wait));
#else   /* VXIPNP */
            CHECK(e1432_read_float64_data(hw, in_chan_list[i], OCTAVE_DATA,
              oct_buf, bands, &trailer, &count));
#endif  /* VXIPNP */
            /* do the average */
            dptr = *rptr++;
            doptr = oct_buf;
            for ( j = 0; j < bands; j++ )
            {
                *dptr++ += *doptr++;
            }
        }
    }

    /* normalize and convert to dB */
    rptr = results;
    for (i = 0; i < in_chans; i++ )
    {
        dptr = *rptr++;
        for ( j = 0; j < bands; j++ )
        {
            *dptr++ = 10 * log10(*dptr/(double)avgs);
        }
    }

    return 0;
}


LIB_STATUS
oct_1_band_meas(LIB_HW hw, LIB_GROUP in_group, LIB_INT in_chans,
  LIB_ID *in_chan_list, double *results, LIB_FLOAT meas_time)
{
    double **one_bin_results;
    double **tmp;
    int i;

    one_bin_results = (double **)malloc(sizeof(double *) * in_chans);

    tmp = one_bin_results;
    for ( i = 0; i < in_chans; i++ )
    {
        *tmp++ = results++;
    }

    CHECK(oct_meas(hw, in_group, in_chans, in_chan_list, 1, one_bin_results,
      meas_time));

    free(one_bin_results);

    return 0;
}


LIB_STATUS
freq_meas(LIB_HW hw, LIB_GROUP in_group, LIB_INT in_chans,
  LIB_ID *in_chan_list, LIB_GROUP src_group, LIB_FLOAT amplitude,
  LIB_FLOAT *freqs, int n_freqs, LIB_FLOAT *band)
{
    int i, j, k;
    LIB_FLOAT *freq_ptr;
    double **cal_data_array;
    double **cal_data_ptr;
    double *oct_data;
    double *crtd_data, *prev_crtd_data;
    double min, max;
    double min_nbw = 0.0, max_nbw = 0.0, *nbw, ref_bw, exact_mid_band;
    double lw_b_edge, up_b_edge;
    double  *int_resp, min_int_resp, max_int_resp;
    LIB_FLOAT src_freq, delta_freq, prev_freq, oct_band, filter_freq, selected_band;
    LIB_FLOAT octave_int_time, override_time, int_time, meas_time;
    LIB_LONG blocksize = BLOCKSIZE_DEF;
    char data_fmt[] = "%9.3f";
    LIB_ENUM octave_mode;

    /* malloc measurement arrays */
    cal_data_array = (double **)malloc(sizeof(double *) * n_freqs);
    cal_data_ptr = cal_data_array;
    for ( i = 0; i < n_freqs; i++ )
    {
        *cal_data_ptr++ = (double *)malloc(sizeof(double) * in_chans);
    }
    oct_data = (double *)malloc(sizeof(double) * in_chans);
    crtd_data = (double *)malloc(sizeof(double) * in_chans);
    prev_crtd_data = (double *)malloc(sizeof(double) * in_chans);
    nbw = (double *)malloc(sizeof(double) * in_chans);
    int_resp = (double *)malloc(sizeof(double) * in_chans);
    /* set initial value of nbw array */
    for ( j = 0; j < in_chans; j++ )
    {
      nbw[j] = 0.0;
      prev_crtd_data[j] = 0.0;
    }

    /* fixed source setup */
    CHECK(e1432_set_active(hw, src_group, CHANNEL_ON));
    CHECK(e1432_set_range(hw, src_group, amplitude));
    CHECK(e1432_set_amp_scale(hw, src_group, 1.0));
    CHECK(e1432_set_ramp_rate(hw, src_group, 0.0));

    if ( verbose || print_raw )
    {
        print_chan_nums(in_chan_list, in_chans, "chans: ", "%9d");
    }

    /* fft measurements (cal values) */
    freq_ptr = freqs;
    cal_data_ptr = cal_data_array;
    for ( i = 0; i < n_freqs; i++ )
    {
        src_freq = *freq_ptr++;

        /* source */
        filter_freq = FILTER_FREQ(src_freq);
        CHECK(e1432_set_filter_freq(hw, src_group, filter_freq));
        CHECK(e1432_set_sine_freq(hw, src_group, src_freq));

        CHECK(fft_meas(hw, in_group, in_chans, in_chan_list, *cal_data_ptr++,
          src_freq, fft_meas_time));
    }

    
    CHECK(e1432_get_octave_mode(hw, in_group, &octave_mode));
    
    exact_mid_band = exact_fm(*band, octave_mode);
    if ( *band > 0 )
    {
      if ( octave_mode == OCTAVE_MODE_FULL ) 
      {
        printf("\nOctave mode = full\n");
        selected_band = octave_band(*band);
        lw_b_edge=exact_mid_band*pow(2,(-1.0/2.0));
        up_b_edge=exact_mid_band*pow(2,(1.0/2.0)); 
      }
      else if ( octave_mode == OCTAVE_MODE_THIRD )
      {  
        printf("\nOctave mode = third\n");
        selected_band = third_oct_band(*band);
        lw_b_edge=exact_mid_band*pow(2,(-1.0/6.0));
        up_b_edge=exact_mid_band*pow(2,(1.0/6.0)); 
      }
      else
      {
         (void) fprintf(stderr, "octave mode(%d) not defined.\n", octave_mode);
         exit(2);
      }	
      printf("Selected band = %.2f\n", selected_band);
      printf("Exact mid band = %.4f\n", exact_mid_band);
      printf("Exact band edges: %.4f to %.4f\n", lw_b_edge, up_b_edge);
      printf("\n Frequency   MinLvl   MaxLvl\n");
    }
    else
    {
      selected_band = *band;
    }

    /* Octave measurements */
    CHECK(e1432_set_octave_meas(hw, in_group, OCTAVE_MEAS_ON));
    CHECK(e1432_get_octave_int_time(hw, in_group, &octave_int_time));
    freq_ptr = freqs;
    cal_data_ptr = cal_data_array;
    for ( i = 0; i < n_freqs; i++ )
    {
        src_freq = *freq_ptr++;
	/* Force meas freq/band to nearest band if non-zero, othewise closest band
	   to source frequency. */
        oct_band = (*band == 0 ? src_freq : selected_band);

        /* source */
        filter_freq = FILTER_FREQ(src_freq);
        CHECK(e1432_set_filter_freq(hw, src_group, filter_freq));
        CHECK(e1432_set_sine_freq(hw, src_group, src_freq));

        /* stop/start bands */
        CHECK(e1432_set_octave_start_freq(hw, in_group, oct_band));
        CHECK(e1432_set_octave_stop_freq(hw, in_group, oct_band));

        /*override octave_int_time and oct_meas_time with -e option*/
        override_time = min_input_cycles / src_freq;
        if (override_time > octave_int_time)
        {
            int_time = override_time;
        }
        else
        {
            int_time = octave_int_time;
        }
        CHECK(e1432_set_octave_int_time(hw, in_group, int_time));
        if (override_time > oct_meas_time)
        {
            meas_time = override_time;
        }
        else
        {
            meas_time = oct_meas_time;
        }
        /*printf("int_time=%g, meas_time=%g\n", int_time, meas_time);*/
                   
        /* settling time */
        CHECK(set_filter_settling_time(hw, in_group, oct_band));

          
        CHECK(oct_1_band_meas(hw, in_group, in_chans, in_chan_list, oct_data,
          meas_time));

        v_subt(oct_data, *cal_data_ptr, crtd_data, in_chans);

        if ( verbose || print_raw )
        {
            printf("\nfreq %.3f:\n", src_freq);
            if ( print_raw )
            {
                printf("meas:  ");
                print_doubles(oct_data, in_chans, data_fmt);
                printf("crtn:  ");
                print_doubles(*cal_data_ptr, in_chans, data_fmt);
                printf("crt'd: ");
            }
            else
            {
                printf("       ");
            }
            print_doubles(crtd_data, in_chans, data_fmt);
            s_min_max(&min, &max, crtd_data, in_chans);
        }
        else
        {
            s_min_max(&min, &max, crtd_data, in_chans);
            printf("%10.3lf %8.3lf %8.3lf\n",
	      src_freq, min, max);
        }
        
	/* compute noise bandwidth with trapezoid rule(channel 1 for now) */
	if ( n_freqs <= 1) 
	{
	  delta_freq = 0.0; /* just do something */
	}
	else if ( i == 0 ) 
	{
	  delta_freq = 0.0; /* just do something */
	}
	else 
	{
	  delta_freq = (src_freq - prev_freq);
	  for ( j = 0; j < in_chans; j++ )
	  {
	    nbw[j] += .5*(pow(10.0, prev_crtd_data[j]/10.0) + pow(10.0, crtd_data[j]/10.0)) * delta_freq; 
	  }
	}
	      
        (void) fflush(NULL);

        /* set previous value of prev_crtd_data */
        for ( j = 0; j < in_chans; j++ )
        {
          prev_crtd_data[j] = crtd_data[j];
        }        
        prev_freq = src_freq;
        cal_data_ptr++;
    }
    
    if ( *band > 0 )
    {
	s_min_max(&min_nbw, &max_nbw, nbw, in_chans);
	
	/*calculate filter integrated response(IEC1260)*/
        ref_bw = ref_nbw(selected_band, octave_mode);
        for ( j = 0; j < in_chans; j++ )
        {
          int_resp[j] = 10.0*log10(nbw[j]/ref_bw);
        } 
        s_min_max(&min_int_resp, &max_int_resp, int_resp, in_chans);
        
	if ( verbose || print_raw )
	{
	  printf("\nmeasured noise BW = \n");
	  print_doubles(nbw, in_chans, data_fmt);
	  printf("\nmeasured integrated response = \n");
	  print_doubles(int_resp, in_chans, data_fmt);
	}
	else
	{
	  printf("\nmeasured noise BW = %.3lf to %.3lf\n", min_nbw, max_nbw);
	  printf("measured integrated response = %.3lf to %.3lf\n", min_int_resp, max_int_resp);
	}
    }

    /* clean up */
    CHECK(e1432_set_octave_meas(hw, in_group, OCTAVE_MEAS_OFF));

    /* free measurement arrays */
    cal_data_ptr = cal_data_array;
    for ( i = 0; i < n_freqs; i++ )
    {
        free(*cal_data_ptr++);
    }
    free(cal_data_array);
    free(oct_data);
    free(crtd_data);
    free(prev_crtd_data);
    free(nbw);

    return 0;
}


LIB_STATUS
amp_lin_meas(LIB_HW hw, LIB_GROUP in_group, LIB_INT in_chans,
  LIB_ID *in_chan_list, LIB_GROUP src_group, LIB_FLOAT freq,
  LIB_FLOAT range, LIB_FLOAT start_amp, LIB_FLOAT amp_step,
  LIB_FLOAT *lim_ptr, int n_limits, int relative_cal)
{
    double *cal_data, *oct_data, *crtd_data, *err;
    LIB_FLOAT filter_freq;
    double amp, range_db;
    LIB_FLOAT amp_scale, lim;
    LIB_FLOAT int_time, meas_time, override_time, octave_int_time;
    LIB_LONG blocksize = BLOCKSIZE_DEF;
    double min, max;
    int lim_count = 0;
    char data_fmt[] = "%8.3f";

    /* malloc measurement arrays */
    cal_data = (double *)malloc(sizeof(double) * in_chans);
    oct_data = (double *)malloc(sizeof(double) * in_chans);
    crtd_data = (double *)malloc(sizeof(double) * in_chans);
    err = (double *)malloc(sizeof(double) * in_chans);

    range_db = 20.0 * log10(range);

    /* source setup */
    CHECK(e1432_set_active(hw, src_group, CHANNEL_ON));
    CHECK(e1432_set_range(hw, src_group, range));
    CHECK(e1432_set_amp_scale(hw, src_group, 1.0));
    CHECK(e1432_set_ramp_rate(hw, src_group, 0.0));
    filter_freq = FILTER_FREQ(freq);
    CHECK(e1432_set_filter_freq(hw, src_group, filter_freq));
    CHECK(e1432_set_sine_freq(hw, src_group, freq));

    /* fft measurement (cal values) (absolute reference) */
    if ( ! relative_cal )
    {
        CHECK(fft_meas(hw, in_group, in_chans, in_chan_list, cal_data, freq,
          fft_meas_time));
    }

    /* Octave measurements */
    CHECK(e1432_set_octave_meas(hw, in_group, OCTAVE_MEAS_ON));
    CHECK(e1432_get_octave_int_time(hw, in_group, &octave_int_time));
    
    /* stop/start bands */
    CHECK(e1432_set_octave_start_freq(hw, in_group, freq));
    CHECK(e1432_set_octave_stop_freq(hw, in_group, freq));

    /*override octave_int_time and oct_meas_time with -e option*/
    override_time = min_input_cycles / freq;
    if (override_time > octave_int_time)
    {
        int_time = override_time;
    }
    else
    {
        int_time = octave_int_time;
    }
    CHECK(e1432_set_octave_int_time(hw, in_group, int_time));
    if (override_time > oct_meas_time)
    {
        meas_time = override_time;
    }
    else
    {
        meas_time = oct_meas_time;
    }
    
    /* settling time */
    CHECK(set_filter_settling_time(hw, in_group, freq));

    if ( relative_cal )
    {
        CHECK(oct_1_band_meas(hw, in_group, in_chans, in_chan_list, cal_data,
          meas_time));
    }

    if (verbose || print_raw)
    {
        printf("\nrange =  %.3f volts peak\n", range);
    }
    
    print_chan_nums(in_chan_list, in_chans, "chans:         ", "%8d");
    printf("     level(limit)\n");

    amp = start_amp;
    for ( ; ; ) /* end of loop is the lim_count >= n_limits break */
    {
        amp_scale = pow(10.0, amp/20.0);
        CHECK(e1432_set_amp_scale(hw, src_group, amp_scale));

        CHECK(oct_1_band_meas(hw, in_group, in_chans, in_chan_list, oct_data,
          meas_time));

        v_subt(oct_data, cal_data, crtd_data, in_chans);
        s_subt(crtd_data, amp, err, in_chans);
        s_min_max(&min, &max, err, in_chans);

        if ( print_raw )
        {
            printf("\namp %.3f dBFS:\n", amp);
            printf("meas:  ");
            print_doubles(oct_data, in_chans, data_fmt);
            printf("crtn:  ");
            print_doubles(cal_data, in_chans, data_fmt);
            printf("crt'd: ");
            print_doubles(crtd_data, in_chans, data_fmt);
            printf("err:   ");
            print_doubles(err, in_chans, data_fmt);
        }
        else
        {
            if  (min < -(*lim_ptr) || max > *lim_ptr)
            {
                printf("**%5.1fdBFS(%4.2f)", amp, *lim_ptr);
                print_doubles(err, in_chans, data_fmt);
            }
            else
            {
                printf("  %5.1fdBFS(%4.2f)", amp, *lim_ptr);
                print_doubles(err, in_chans, data_fmt);
            }
        }
        lim_ptr++;
        lim_count++;
        if ( lim_count >= n_limits )
        {
#ifdef PRINT_PARMS
            if ( ! verbose )
            {
                CHECK(print_parms(hw, in_group));
            }
#endif
            break;
        }
        (void) fflush(NULL);

        amp -= amp_step;
    }

    /* clean up */
    CHECK(e1432_set_octave_meas(hw, in_group, OCTAVE_MEAS_OFF));

    /* free measurement arrays */
    free(cal_data);
    free(oct_data);
    free(crtd_data);
    free(err);

    return 0;
}


LIB_STATUS
decay_meas(LIB_HW hw, LIB_GROUP in_group, LIB_INT in_chans,
  LIB_ID *in_chan_list, LIB_GROUP src_group, LIB_FLOAT amplitude,
  LIB_FLOAT freq, LIB_FLOAT *lim_ptr, int n_limits)
{
    LIB_FLOAT filter_settling_time;
    LIB_FLOAT octave_time_step, octave_time_const;
    double *data, *prev_data, *delta, *delta_err, *tmp;
    double min, max, dbfs;
    double full_scale, ideal_delta;
    int i;
    int bot_count;
    LIB_FLOAT filter_freq = FILTER_FREQ(freq);
    char data_fmt[] = "%9.3f";
    int lim_count = 0;

    data = (double *)malloc(sizeof(double) * in_chans);
    prev_data = (double *)malloc(sizeof(double) * in_chans);
    delta = (double *)malloc(sizeof(double) * in_chans);
    delta_err = (double *)malloc(sizeof(double) * in_chans);

    /* pick up set values */
    CHECK(e1432_get_octave_time_step(hw, in_group, &octave_time_step));
    CHECK(e1432_get_octave_time_const(hw, in_group, &octave_time_const));

    full_scale = 20 * log10(amplitude/sqrt(2));
    ideal_delta = 10 * log10(exp(-octave_time_step/octave_time_const));
    if ( verbose )
    {
        (void) printf("ideal delta = %.3f\n", ideal_delta);
    }

    /* source */
    CHECK(e1432_set_active(hw, src_group, CHANNEL_ON));
    CHECK(e1432_set_range(hw, src_group, amplitude));
    CHECK(e1432_set_amp_scale(hw, src_group, 1.0));
    CHECK(e1432_set_ramp_rate(hw, src_group, 0.0));
    CHECK(e1432_set_filter_freq(hw, src_group, filter_freq));
    CHECK(e1432_set_sine_freq(hw, src_group, freq));

    CHECK(e1432_set_octave_meas(hw, in_group, OCTAVE_MEAS_ON));

    /* Octave start/stop band */
    CHECK(e1432_set_octave_start_freq(hw, in_group, freq));
    CHECK(e1432_set_octave_stop_freq(hw, in_group, freq));

    /* set filter_settling_time .3 normal to get close */
    CHECK(set_filter_settling_time(hw, in_group, freq));
    CHECK(e1432_get_filter_settling_time(hw, in_group, &filter_settling_time));
    filter_settling_time *= .3;
    CHECK(e1432_set_filter_settling_time(hw, in_group, filter_settling_time));

    /* enable only Octave data */
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_TIME,
      ENABLE_OFF));
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_FREQ,
      ENABLE_OFF));
    CHECK(e1432_set_enable(hw, in_group, ENABLE_TYPE_OCTAVE,
      ENABLE_ON));

    /* Start measurement */
    CHECK(e1432_init_measure(hw, in_group));

    if ( print_raw )
    {
        (void) printf("wait BA\n");
    }
    /* wait for first data before turning source off */
    CHECK(wait_block_available(hw, in_group, 0));

    if ( print_raw )
    {
        (void) printf("turning source off\n");
    }
    /* turn source off */
    CHECK(e1432_set_amp_scale(hw, src_group, 0.0));

    /* read the initial values */
    CHECK(read_f64_data(hw, in_chan_list, in_chans,OCTAVE_DATA, data, 1));
    if ( print_raw )
    {
        (void) printf("\ninitial: ");
        (void) print_doubles(data, in_chans, "%8.3f");
    }

    /* wait until */
    do
    {
        tmp = data; data = prev_data; prev_data = tmp;  /* data => prev_data */
        CHECK(wait_block_available(hw, in_group, 0));
        CHECK(read_f64_data(hw, in_chan_list, in_chans, OCTAVE_DATA,
          data, 1));
        v_subt(data, prev_data, delta, in_chans);
        s_min_max(&min, &max, delta, in_chans);
        if ( print_raw )
        {
            (void) printf("\nwaiting: ");
            (void) print_doubles(data, in_chans, "%8.3f");
        }
    }
    while ( max > 0 );  /* until all deltas are negative */

#define DO_TOSS
#ifdef DO_TOSS
    /* toss about filter_settling_time worth to be sure it is in decay mode */
    for ( i = 0; i < .25 * filter_settling_time / octave_time_step; i++ )
    {
        tmp = data; data = prev_data; prev_data = tmp;  /* data => prev_data */
        CHECK(wait_block_available(hw, in_group, 0));
        CHECK(read_f64_data(hw, in_chan_list, in_chans, OCTAVE_DATA, data, 1));
        if ( print_raw )
        {
            (void) printf("\ntossing: ");
            (void) print_doubles(data, in_chans, "%8.3f");
        }
    }
#endif

    /* read until hit bottom for awhile */
    bot_count = 5;
    do
    {
        tmp = data; data = prev_data; prev_data = tmp;  /* data => prev_data */
        CHECK(wait_block_available(hw, in_group, 0));
        CHECK(read_f64_data(hw, in_chan_list, in_chans, OCTAVE_DATA, data, 1));
        v_subt(data, prev_data, delta, in_chans);
        s_min_max(&min, &max, delta, in_chans);
        if ( max > 0 ) bot_count--;
        s_subt(data, full_scale, prev_data, in_chans);  /* dbFs in prev_data */
        s_avg(&dbfs, prev_data, in_chans);
        /* compute error in delta */
        s_div(delta, ideal_delta, delta_err, in_chans);
        s_subt(delta_err, 1.0, delta_err, in_chans);
        s_mult(delta_err, 100, delta_err, in_chans);
        if ( print_raw )
        {
            (void) printf("\nmeas:  ");
            print_doubles(data, in_chans, data_fmt);
            (void) printf("dBFS:  ");
            print_doubles(prev_data, in_chans, data_fmt);
            (void) printf("delta: ");
            print_doubles(delta, in_chans, data_fmt);
            (void) printf("%% err: ");
            print_doubles(delta_err, in_chans, data_fmt);
        }

        /* check percent error in delta against limits */
        s_min_max(&min, &max, delta_err, in_chans);
        while ( lim_count < n_limits && (min < -(*lim_ptr) || max > *lim_ptr) )
        {
            if ( -min > max ) max = -min;
            (void) printf("%.2lf%% decay time constant error at %.2lf dBFS",
              max, dbfs);
            (void) printf(" (< %.2lf%% error at %.2lf dBFS)\n",
              *lim_ptr, dbfs - ideal_delta);
            lim_ptr++;
            lim_count++;
        }
        if ( lim_count >= n_limits )
        {
#ifdef PRINT_PARMS
            if ( ! verbose )
            {
                (void) printf("  ideal decay delta = %.3f\n", ideal_delta);
                CHECK(print_parms(hw, in_group));
            }
#endif
            break;
        }
        (void) fflush(NULL);
    }
    while ( bot_count > 0 );

    free(data);
    free(prev_data);
    free(delta);
    free(delta_err);

    return 0;
}


LIB_STATUS
multi_bin_meas(LIB_HW hw, LIB_GROUP in_group, int filt_compare,
  LIB_INT in_chans, LIB_ID *in_chan_list, LIB_GROUP src_group,
  LIB_FLOAT amplitude, LIB_FLOAT freq, LIB_FLOAT octave_start_freq,
  LIB_FLOAT octave_stop_freq)
{
    LIB_FLOAT filter_freq = FILTER_FREQ(freq);
    LIB_LONG bands;
    double **all_chans;
    double **dpptr;
    double *cal_data, *avg, *pdelta, *ndelta;
    double *ref, *ref_min, *ref_max;
    double *delta, *delta_min, *delta_max;
    double *band_sums;
    LIB_FLOAT range;
    double full_scale, min, max;
    LIB_FLOAT band_step;
    LIB_ENUM octave_mode;
    int i,j;
    char data_fmt[] = "%9.3f";
    LIB_FLOAT int_time, meas_time, octave_int_time, override_time;

    /* malloc cal array */
    cal_data = (double *)malloc(sizeof(double) * in_chans);

    /* malloc band_sum array and intialize*/
    band_sums = (double *)malloc(sizeof(double) * in_chans);
    for ( i = 0; i < in_chans; i++ )
    {
        band_sums[i] = 0.0;
    }

    CHECK(e1432_get_range(hw, in_group, &range));
    full_scale = 20 * log10(sqrt(.5) * range);

    CHECK(e1432_get_octave_mode(hw, in_group, &octave_mode));

    /* source */
    CHECK(e1432_set_active(hw, src_group, CHANNEL_ON));
    CHECK(e1432_set_range(hw, src_group, amplitude));
    CHECK(e1432_set_amp_scale(hw, src_group, 1.0));
    CHECK(e1432_set_ramp_rate(hw, src_group, 0.0));

    CHECK(e1432_set_filter_freq(hw, src_group, filter_freq));
    CHECK(e1432_set_sine_freq(hw, src_group, freq));

    /* generate a relative correction for each channel */
    CHECK(fft_meas(hw, in_group, in_chans, in_chan_list, cal_data, freq,
      fft_meas_time));
    s_subt(cal_data, 20 * log10(sqrt(.5) * amplitude), cal_data, in_chans);

    CHECK(e1432_set_octave_meas(hw, in_group, OCTAVE_MEAS_ON));
    CHECK(e1432_get_octave_int_time(hw, in_group, &octave_int_time));
    
    /*override octave_int_time and oct_meas_time with -e option*/
    override_time = min_input_cycles / freq;
    if (override_time > octave_int_time)
    {
        int_time = override_time;
    }
    else
    {
        int_time = octave_int_time;
    }
    CHECK(e1432_set_octave_int_time(hw, in_group, int_time));
    if (override_time > oct_meas_time)
    {
        meas_time = override_time;
    }
    else
    {
        meas_time = oct_meas_time;
    }
    
    /* stop/start bands */
    CHECK(e1432_set_octave_start_freq(hw, in_group, octave_start_freq));
    CHECK(e1432_set_octave_stop_freq(hw, in_group, octave_stop_freq));
    CHECK(e1432_get_octave_start_freq(hw, in_group, &octave_start_freq));
    CHECK(e1432_get_octave_stop_freq(hw, in_group, &octave_stop_freq));
    bands = oct_bands(hw, in_group, octave_start_freq, octave_stop_freq);
    band_step = oct_band_step(hw, in_group);

    /* malloc the measurement arrays, now that number of bands is known */
    all_chans = (double **)malloc(sizeof(double) * in_chans);
    dpptr = all_chans;
    for ( i = 0; i < in_chans; i++ )
    {
        *dpptr++ = (double *)malloc(sizeof(double) * bands);
    }
    avg = (double *)malloc(sizeof(double) * bands);
    pdelta = (double *)malloc(sizeof(double) * bands);
    ndelta = (double *)malloc(sizeof(double) * bands);
    ref = (double *)malloc(sizeof(double) * bands);
    ref_min = (double *)malloc(sizeof(double) * bands);
    ref_max = (double *)malloc(sizeof(double) * bands);
    delta = (double *)malloc(sizeof(double) * bands);
    delta_min = (double *)malloc(sizeof(double) * bands);
    delta_max = (double *)malloc(sizeof(double) * bands);

    /* calculate the value expected in each bin */
    if ( filt_compare )
    {
        gen_ideal(freq, 1.0, octave_start_freq, octave_mode, ref, bands);
        gen_ideal(freq, QD_MAX, octave_start_freq, octave_mode, ref_min, bands);
        gen_ideal(freq, QD_MIN, octave_start_freq, octave_mode, ref_max, bands);
        if ( verbose )
        {
            print_bands("\nideal:", ref, bands, octave_start_freq, band_step);
            print_bands("\nmin:", ref_min, bands, octave_start_freq, band_step);
            print_bands("\nmax:", ref_max, bands, octave_start_freq, band_step);
	    printf("\n");
        }
    }

    /* settling time */
    CHECK(set_filter_settling_time(hw, in_group, octave_start_freq));

    CHECK(oct_meas(hw, in_group, in_chans, in_chan_list, bands, all_chans,
      meas_time));

    /* adjust each band by the cal values */
    dpptr = all_chans;
    for ( i = 0; i < in_chans; i++ )
    {
        s_subt(*dpptr, cal_data[i], *dpptr, bands);
	dpptr++;
    }

    /* a little hokey, averaging dB's, but reasonable when nearly the same */
    v_avg(avg, all_chans, in_chans, bands);

    /* compute the max/min deltas from avg */
    v_min_max(ndelta, pdelta, all_chans, in_chans, bands);
    v_subt(ndelta, avg, ndelta, bands);
    v_subt(pdelta, avg, pdelta, bands);

    /* correct average for full scale */
    s_subt(avg, full_scale, avg, bands);

    print_bands("average of channels:",
      avg, bands, octave_start_freq, band_step);
    
    /*now rms sum levels in all bands and print out*/
    for ( i = 0; i < in_chans; i++ )
    {
        for ( j = 0; j < bands; j++ )
        {
           band_sums[i] = band_sums[i] + pow(10.0, (all_chans[i][j])/10.0);
        }
        /*convert to db*/
        band_sums[i] = 10.0*log10((band_sums[i]));
        /*correct for full scale*/
        band_sums[i] = band_sums[i] - full_scale;
    }
    /*get min and max*/
    s_min_max(&min, &max, band_sums, in_chans);
    printf("rms sum of output signals:\n");
    if (verbose || print_raw)
    {
        print_doubles(band_sums, in_chans, data_fmt);
    }
    else
    {
        printf("    min=%g, max=%g\n\n", min, max);
    }
        
    
    if ( verbose )
    {
        print_bands("\npositive deltas:", pdelta, bands, octave_start_freq,
          band_step);
        print_bands("\nnegative deltas:", ndelta, bands, octave_start_freq,
          band_step);
	printf("\n");
    }

    if ( filt_compare )
    {
        v_subt(avg, ref, delta, bands);
        v_subt(avg, ref_min, delta_min, bands);
        v_subt(avg, ref_max, delta_max, bands);
        print_bands("delta from ideal:",
          delta, bands, octave_start_freq, band_step);
        print_bands("delta from min:",
          delta_min, bands, octave_start_freq, band_step);
        print_bands("delta from max:",
          delta_max, bands, octave_start_freq, band_step);
    }

    dpptr = all_chans;
    for ( i = 0; i < in_chans; i++ )
    {
        free(*dpptr++);
    }
    free(band_sums);
    free(all_chans);
    free(avg);
    free(pdelta);
    free(ndelta);
    free(ref);
    free(ref_min);
    free(ref_max);
    free(delta);
    free(delta_min);
    free(delta_max);
    free(cal_data);

    return 0;
}



int
main(int argn, char **argv)
{
    /* command line vars */
    int opt;               /* really char returned */

    /* boiler plate vars */
    LIB_INT modules = 1;
    int la_override = 0;
    LIB_INT laddr[MAX_LAS] = { 8 };  /* default logical address */
    LIB_HW hw;
    LIB_INT in_chans = 0;
    LIB_ID in_chan_list[MAX_CHANS];
    LIB_GROUP in_group;
#ifdef  VXIPNP
    ViChar interface[] = "VXI0";
    ViChar openstring[200], tempstring[200];
    ViInt32 hwinfo[27*MAX_LAS];
#else   /* VXIPNP */
    struct e1432_hwconfig hwconfig[MAX_LAS];
#endif  /* VXIPNP */

    /* general use */
    int i;
    LIB_LONG loop;
    LIB_FLOAT src_amp =  0.0;

    /* general api vars */
    LIB_LONG blocksize = BLOCKSIZE_DEF;
    LIB_ENUM data_mode = DATA_MODE_OVERLAP_BLOCK;
    LIB_ENUM data_size = DATA_SIZE_DEF;
    LIB_ENUM weighting = WEIGHTING_OFF;
    LIB_FLOAT range = 1.0;
    LIB_ENUM coupling = COUPLING_DC;
    LIB_FLOAT coupling_freq = 1.0;
    LIB_LONG test_loops = 1;
    
    /* octave api vars */
    LIB_ENUM octave_mode = OCTAVE_MODE_DEF;
    LIB_ENUM octave_avg_mode = OCTAVE_AVG_MODE_DEF;
    LIB_ENUM octave_hold_mode = OCTAVE_HOLD_MODE_DEF;
    LIB_FLOAT octave_int_time = OCTAVE_INT_TIME_DEF;
    LIB_FLOAT octave_time_const = OCTAVE_TIME_CONST_DEF;
    LIB_FLOAT octave_time_step = OCTAVE_TIME_STEP_DEF;

    /* source */
    LIB_ID src_chan_list[MAX_SRCS];
    LIB_GROUP src_group;
    int src_chans = 0;

    /* freq test values */
    LIB_FLOAT freqs[MAX_FREQ_PTS];
    int n_freqs = 0;

    /* amplitude linearity test values */
    LIB_FLOAT amp_lin_lims[MAX_AMP_LIN_PTS];
    int n_amp_lin_lims = 0;
    int relative_cal = 0;  /* to use FS Octave meas for amp lin ref */

    /* exponential decay test values */
    LIB_FLOAT decay_lims[MAX_DECAY_PTS];
    int n_decay_lims = 0;

    LIB_FLOAT band_freqs[] = {0.0, 0.0};
    int filt_compare = 0;

    /* tests to do */
    int do_freq_test = 0;
    int do_amp_lin_test = 0;
    int do_decay_test = 0;
    int do_band_test = 0;

    while ( ( opt = getopt(argn, argv, VALID_OPTS) ) != EOF )
    {
        switch (opt)
        {
        case 'a':
            octave_avg_mode = get_enum(optarg, octave_avg_mode_enums,
              octave_avg_mode_strs, "octave_avg_mode");
            break;
        case 'b':
            blocksize = get_long(optarg, "time blocksize");
            break;
        case 'c':
            octave_time_const = get_float(optarg, "octave_time_const");
            break;
        case 'd':
            max_cal_meas_time = get_float(optarg, "max_cal_meas_time");
            break;
        case 'e':
            min_input_cycles = get_float(optarg, "min_input_cycles");
            break;
        case 'f':
            test_freq = get_float(optarg, "test frequency");
            break;
        case 'h':
            octave_hold_mode = get_enum(optarg, octave_hold_mode_enums,
              octave_hold_mode_strs, "octave_hold_mode");
            break;
        case 'i':
            octave_int_time = get_float(optarg, "octave_int_time");
            break;
        case 'l':
            /*amp_lin_lims correspond to amplitude steps*/
            /*   defined by amp_step(-S option) and amp_start(-T option)*/
            /*   there are as many amplitude steps as there are amp_lin_lims*/
            n_amp_lin_lims = get_float_array(optarg,
               "amplitude linearity error limits", amp_lin_lims,
               MAX_AMP_LIN_PTS);
            do_amp_lin_test = 1;
            break;
        case 'm':
            data_mode =
              get_enum(optarg, data_mode_enums, data_mode_strs, "data mode");
            break;
        case 'n':
            test_loops = get_long(optarg, "test loops");
            break;
        case 'o':
            octave_mode = get_enum(optarg, octave_mode_enums, octave_mode_strs,
              "octave_mode");
            break;
        case 'p':
            /*IEC 1260 format for amplitude linearity testing*/
            /*must be specified after -S and -T so amp_lin_lims can be filled*/
	    {
                LIB_FLOAT vars[4], lin_error, lin_range, max_error, max_range;
		i = get_float_array(optarg, "lin_error/lin_range/max_error/max_range", vars, 4);
		if ( i != 4 )
		{
                    (void) fprintf(stderr,
		      "in_error, lin_range, max_error, max_range -p option\n");
                    exit(2);
		}
		lin_error = vars[0];
		lin_range = vars[1];
		max_error = vars[2];
		max_range = vars[3];
		if ( lin_error < 0 || lin_range < 0 || max_error < lin_error || max_range < lin_range )
		{
                    (void) fprintf(stderr, "value err(s) for -G option parms:\n"
		      "  lin_error = %g, lin_range = %g, max_error = %g, max_range = %g\n",
		      lin_error, lin_range, max_error, max_range);
                    exit(2);
		}
                src_amp = amp_start;
                i = 0;
                /*now fill up amp_lin_lims[]*/
                while ( src_amp >= -1.0*max_range)
                {
                    if (src_amp >= -1.0*lin_range)
                    {
                        amp_lin_lims[i] = lin_error;
                    }
                    else
                    {
                        amp_lin_lims[i] = max_error;
                    }
                    i++;
                    src_amp = src_amp - amp_step;
                    /*printf("src_amp = %g\n", src_amp); */                  
                } 
            }
            n_amp_lin_lims = i;
	    do_amp_lin_test = 1;
            break;     
        case 'r':
            range = get_float(optarg, "range");
            break;
        case 's':
            data_size =
              get_enum(optarg, data_size_enums, data_size_strs, "data size");
            break;
        case 't':
            octave_time_step = get_float(optarg, "octave_time_step");
            break;    
        case 'v':
            verbose = 1;
            break;
        case 'z':
            filt_compare = 1;
            break;
        case 'A':
            coupling = COUPLING_AC;
            coupling_freq = get_float(optarg, "coupling frequency");
            break;
        case 'B':
            i = get_float_array(optarg, "start/stop bands", band_freqs, 2);
            if ( i < 1 )
            {
                (void) fprintf(stderr,
                  "expected 2 bands for start/stop bands or one band"
		  "for fixed band frequency response measurement\n");
                exit(2);
            }
            do_band_test = (i == 2);
            break;
        case 'C':
            fft_meas_time = get_float(optarg, "cal meas time");
            break;
        case 'D':
            n_decay_lims = get_float_array(optarg, "exp decay points",
              decay_lims, MAX_DECAY_PTS);
            do_decay_test = 1;
            break;
        case 'F':
            n_freqs =
              get_float_array(optarg, "frequencies", freqs, MAX_FREQ_PTS);
            do_freq_test = 1;
            break;
        case 'G':
	    {
                float vars[3], start, stop, delta, f, *fp;
		i = get_float_array(optarg, "start/stop/delta", vars, 3);
		if ( i != 3 )
		{
                    (void) fprintf(stderr,
		      "expected start stop delta for -G option\n");
                    exit(2);
		}
		start = vars[0];
		stop = vars[1];
		delta = vars[2];
		if ( stop < start || start < 0 || stop <= 0 || delta <= 0 )
		{
                    (void) fprintf(stderr, "value err(s) for -G option parms:\n"
		      "  start = %g, stop = %g, delta = %g\n",
		      start, stop, delta);
                    exit(2);
		}
		n_freqs = (stop - start) / delta + 1.5;
		if ( n_freqs > MAX_FREQ_PTS )
		{
                    (void) fprintf(stderr, "number of points (%d) for -G"
		      "option exceeds maximum of %d\n", n_freqs, MAX_FREQ_PTS);
                    exit(2);
		}
		/* fill up the freqs array with the points */
                fp = freqs;
		f = start;
		for ( i = 0; i < n_freqs; i++ )
		{
		    *fp++ = f;
		    f += delta;
		}
                do_freq_test = 1;
	    }
            break;
        case 'H':
	    {
                float vars[3], fm, S, N, f, *fp, exp, x, b;
                i = get_float_array(optarg, "fm/S/N", vars, 3);
		if ( i != 3 )
		{
                    (void) fprintf(stderr,
		      "expected fm S N for -H option\n");
                    exit(2);
		}
		fm = vars[0];
		S = vars[1];  
		N = vars[2]; 
		if ( fm <= 0 || S <= 0 || N <= 0 )
		{
                    (void) fprintf(stderr, "value err(s) for -H option parms:\n"
		      "  fm = %g, S = %g, N = %g\n",
		      fm, S, N);
                    exit(2);
		}
		n_freqs = 2*N + 1;
		if ( n_freqs > MAX_FREQ_PTS )
		{
                    (void) fprintf(stderr, "number of points (%d) for -H"
		      "option exceeds maximum of %d\n", n_freqs, MAX_FREQ_PTS);
                    exit(2);
		}
		/* fill up the freqs array with the points */
                
		if ( octave_mode == OCTAVE_MODE_FULL ) b=1.;
                else if ( octave_mode == OCTAVE_MODE_THIRD ) b=3.;
                else
                {
                    (void) fprintf(stderr, "octave mode(%d) not defined.\n", octave_mode);
		    exit(2);
                }
                fp=freqs;
                n_freqs=0;
                for ( x = -1.*N; x <= N; x++ )
                {
		    exp=x/(b*S);
		    f = fm*pow(2,exp);
		    if (f <= SRC_FILTER_FREQ_MAX)
		    {
		      *fp = f;
		      fp++;
		      n_freqs++;
		    }
		}
		do_freq_test = 1;
	    }
            break;
        case 'L':
            if ( ! la_override )
            {
                la_override = 1;
                modules = 0;
            }
            laddr[modules++] = get_int(optarg, "logical address");
            break;
        case 'M':
            oct_meas_time = get_float(optarg, "oct_meas_time ");
            break;
        case 'R':
            print_raw = 1;
            verbose = 1;
            break;
        case 'S':
            amp_step = get_float(optarg, "amplitude_step");
            break;
        case 'T':
            amp_start = get_float(optarg, "amplitude_start");
            break;
        case 'U':
            relative_cal = 1;
            break;
        default:
            print_usage(argv[0]);
            exit(2);
        }
    }


#ifdef  VXIPNP
    /* build up the init string */
    strcpy(openstring, interface);
    strcat(openstring, "::");
    for ( i = 0; i < modules; i++ )
    {
        sprintf(tempstring, "%d", laddr[i]);
        strcat(openstring, tempstring);
        if ( i > 0 ) strcat(openstring, ",");
    }

    CHECK(hpe1432_init(openstring, 0, 1, &hw));
/* FIX THIS !!! */
CHECK(hpe1432_init("VXI0::8::INSTR", 0, 1, &hw));

    CHECK(hpe1432_setTraceLevel(hw, 0));

    CHECK(hpe1432_getHWConfig(hw, modules, laddr, hwinfo));
    for ( i = 0; i < modules; i++ )
    {
        if ( (hwinfo[HWCONF_SPACE * i + HWCONF_ID] & HWCONF_ID_OCT) == 0 )
        {
            (void) fprintf(stderr, "Octave option not present at LA %d\n",
              laddr[i]);
            exit(1);
        }
        in_chans += hwinfo[HWCONF_SPACE * i + HWCONF_IN_CHANS];
        src_chans += hwinfo[HWCONF_SPACE * i + HWCONF_SRC_CHANS];
    }

#else   /* VXIPNP */

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(modules, laddr, &hw));

    CHECK(e1432_get_hwconfig(modules, laddr, hwconfig));

    for ( i = 0; i < modules; i++ )
    {
        if ( ! hwconfig[i].oct_present)
        {
            (void) fprintf(stderr, "Octave option not present at LA %d\n",
              laddr[i]);
            exit(1);
        }
        in_chans += hwconfig[i].input_chans;
        src_chans += hwconfig[i].source_chans;
    }
#endif  /* VXIPNP */

    /* set up in_chan_list, in_group */
    if ( in_chans > MAX_CHANS )
    {
        (void) fprintf(stderr,
          "Warning: %d channels found, only %d will be used\n",
          in_chans, MAX_CHANS);
        in_chans = MAX_CHANS;
    }
    else if ( in_chans <= 0 )
    {
        (void) fprintf(stderr, "Warning: no input channels found\n");
    }
    else if ( verbose )  printf("%d channels\n", in_chans);

    /* Create channel group */
    for (i = 0; i < in_chans; i++)
    {
        in_chan_list[i] = E1432_INPUT_CHAN(i+1);
    }
#ifdef  VXIPNP
    CHECK(hpe1432_createChannelGroup(hw, in_chans, in_chan_list, &in_group));
#else   /* VXIPNP */
    in_group = e1432_create_channel_group(hw, in_chans, in_chan_list);
#endif  /* VXIPNP */
    if (in_group >= 0)
    {
        DEBUG((void) printf("e1432_create_channel_group in_group return %d\n",
                            in_group));
        exit(1);
    }

    /* create source group, if needed and found */
    if ( src_chans < 1 )
    {
        (void) fprintf(stderr, "Warning, no source channels found\n");
        exit(1);
    }
    if ( src_chans > MAX_SRCS )
    {
        (void) fprintf(stderr, "Warning: %d source channels found,"
          " only %d will be used\n", src_chans, MAX_SRCS);
        src_chans = MAX_SRCS;
    }
    for (i = 0; i < src_chans; i++)
    {
        src_chan_list[i] =  E1432_SOURCE_CHAN(i + 1);
    }
#ifdef  VXIPNP
    CHECK(hpe1432_createChannelGroup(hw, src_chans, src_chan_list, &src_group));
#else   /* VXIPNP */
    src_group = e1432_create_channel_group(hw, src_chans, src_chan_list);
#endif  /* VXIPNP */
    if ( src_group >= 0 )
    {
        (void) fprintf(stderr, "e1432_create_channel_group in_group"
          " return %d\n", src_group);
        exit(1);
    }

    /* Initialize general hardware things */
    CHECK(e1432_set_analog_input(hw, in_group, INPUT_MODE_VOLT,
      INPUT_HIGH_NORMAL, ANTI_ALIAS_ANALOG_ON,
      COUPLING_DC, range));
    CHECK(e1432_set_weighting(hw, in_group, weighting));
    CHECK(e1432_set_coupling(hw, in_group, coupling));
    CHECK(e1432_set_coupling_freq(hw, in_group, coupling_freq));
    CHECK(e1432_set_data_size(hw, in_group, data_size));
    CHECK(e1432_set_blocksize(hw, in_group, blocksize));
    CHECK(e1432_set_clock_freq(hw, in_group, SAMPLE_FREQ));
    CHECK(e1432_set_data_mode(hw, in_group, data_mode));

    CHECK(e1432_set_octave_mode(hw, in_group, octave_mode));
    CHECK(e1432_set_octave_avg_mode(hw, in_group, octave_avg_mode));
    CHECK(e1432_set_octave_hold_mode(hw, in_group, octave_hold_mode));
    CHECK(e1432_set_octave_int_time(hw, in_group, octave_int_time));
    CHECK(e1432_set_octave_time_const(hw, in_group, octave_time_const));
    CHECK(e1432_set_octave_time_step(hw, in_group, octave_time_step));

    if ( verbose )
    {
        CHECK(print_parms(hw, in_group));
    }

    for ( loop = 0; loop < test_loops; loop++ )
    {
        if ( do_freq_test )
        {
            CHECK(freq_meas(hw, in_group, in_chans, in_chan_list, src_group,
              range, freqs, n_freqs, band_freqs));
        }
        if ( do_amp_lin_test )
        {
            CHECK(amp_lin_meas(hw, in_group, in_chans, in_chan_list, src_group,
              test_freq, range, amp_start, amp_step, amp_lin_lims,
	      n_amp_lin_lims, relative_cal));
        }
        if ( do_decay_test )
        {
            CHECK(decay_meas(hw, in_group, in_chans, in_chan_list, src_group,
             range, test_freq, decay_lims, n_decay_lims));
        }
        if ( do_band_test )
        {
            CHECK(multi_bin_meas(hw, in_group, filt_compare, in_chans,
	      in_chan_list, src_group, range, test_freq, band_freqs[0],
	      band_freqs[1]));
        }
    }

    return 0;
}
